/************************************************************************
* \file: trace_be.c
*
* \version: $Id: trace_be.c,v 1.30 2012/05/28 10:27:32 jayanth.mc Exp $
*
* This file implements the basic initialization and uninitialization for
* Trace component.
*
* \component: Gen2 Trace
*
* \author B. Das         bhaskar.das(o)in.bosch.com
*         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
* \modified: Arun V      For Linux Trace
* \modified: Arun V      For usable in TK Environment        
* \copyright: (c) 2003 - 2009 ADIT
*
***********************************************************************/

#ifndef TRACE_DLT_EXIST

#include "trace_base.h"
#include <sys/prctl.h>

#include "trace_be_common.h"
#ifndef TRACE_ALD_INTERFACE_ENABLED
/* semaphore e for process termination */
    IMPORT sem_t g_semExit;
#else
    #include <sys/eventfd.h>
    IMPORT int event_fd;
#endif

/* ptr to shared memory created by Trace FE */
EXPORT TRACE_mgr* g_TRACE_mgr = NULL;



/*Private declarations*/
#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1

/**
* Initiate IO device switching
*
* \parm mgr Pointer to Trace manager
*
* return None
*/
EXPORT void TRACE_chg_iodev(TRACE_mgr* mgr)
{
    TRACE_iodev iodev =
        (TRACE_iodev)mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_PORT_OFFSET];

    if((iodev >= TRACE_UART) && (iodev < TRACE_MAX_IO_DEV))
    {
        if(iodev != mgr->sh->trc.d_cnfg.actv_chan)
        {
            mgr->sh->trc.d_cnfg.swi_chan = iodev;
            /*  wait untill IO is switched */
            pthread_mutex_lock(&mgr->ctrl.condWait.condMutex);
            TRACE_set_flag(&mgr->sh->evt_id[EVT_SWI_CHNL], TRACE_INI_CHNL_SWTCH);
			mgr->ctrl.cmd_info.tsk_id.tid = pthread_self();
			if(!pthread_equal(mgr->ctrl.tsk_id[TRACE_APPL_CMD_TASK].tid, mgr->ctrl.cmd_info.tsk_id.tid))
			{
               pthread_cond_wait(&mgr->ctrl.condWait.condVariable, &mgr->ctrl.condWait.condMutex);
			}
            pthread_mutex_unlock(&mgr->ctrl.condWait.condMutex);
        }
    }
    else
    {
        /* Fix me: Traceout Invalid physical IO switch request */
    }
}


/**
* Trace Read Task
*
* Read incomming trace data from physical layer and route to applications.
*
* \param  stacd   Unused
*         exinf   Pointer to trace manager
*
* \return None
*/
LOCAL VP TRACE_rd_task(VP exinf);
LOCAL VP TRACE_rd_task(VP exinf)
{
    TRACE_mgr* mgr      = (TRACE_mgr*) exinf;
    ER         rc       = E_OK;

    TRACE_flagAttr Attr = {TRACE_WAI_RD_CTRL, TWF_ORW | TWF_BITCLR,
                           0, TMO_FEVR};
        
    prctl(PR_SET_NAME, "TRACE_Read", 0, 0, 0);


    /*Clean up handler for thread clean up stack*/
    pthread_cleanup_push(TRACE_cleanup_hdlr, mgr);
    (void)TRACE_wait_flag(&mgr->sh->evt_id[EVT_COMMON], &Attr);

    rc = (g_TRACE_fnctbl.io_rd_epilog)(mgr);
    while(E_OK <= rc)
    {
        /* test if we are supposed to cancel */
        pthread_testcancel();

        /* Wait for data from TTFis */	
        rc = (g_TRACE_fnctbl.io_rd[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);

        if(TRUE == mgr->ctrl.cmd_info.rd_wait) /* switch task is processing switch request*/
        {
          TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Read task rc =%d\n", rc);
          /* Wake up switch task to proceed with switching */
          pthread_mutex_lock(&mgr->ctrl.condSleep.condMutex);
          pthread_cond_signal(&mgr->ctrl.condSleep.condVariable);
          pthread_mutex_unlock(&mgr->ctrl.condSleep.condMutex);

          /* RD task should wait for the IO switch to complete */
          pthread_mutex_lock(&mgr->ctrl.condWait.condMutex);
          pthread_cond_wait(&mgr->ctrl.condWait.condVariable, &mgr->ctrl.condWait.condMutex);
          pthread_mutex_unlock(&mgr->ctrl.condWait.condMutex);
        }
    }
    pthread_cleanup_pop(1);
    return NULL;
}

/**
* Trace Switch Task
*
* Switch IO device based on user request originating from TTFis
*
* \param  stacd   Unused
*         exinf   Pointer to trace manager
*
* \return None
*/
LOCAL VP TRACE_switch_task(VP exinf);
LOCAL VP TRACE_switch_task(VP exinf)
{
    TRACE_mgr* mgr           = (TRACE_mgr*) exinf;
    ER         rc            = E_OK; 
    volatile S32 stage       = 0;

    TRACE_status_msg msg;
    volatile U8    bContinue = TRUE;
    TRACE_flagAttr   Attr    = {TRACE_INI_CHNL_SWTCH, TWF_ORW | TWF_BITCLR, 
                                0, TMO_FEVR};

/*SWGII-6394:Name of the thread*/
    prctl(PR_SET_NAME, "TRACE_Switch", 0, 0, 0);
    memset(&msg, 0, sizeof(TRACE_status_msg));
    pthread_cleanup_push(TRACE_cleanup_hdlr, mgr);

/* PRQA: QAC Message 3201 : All the cases would be executed  based on stage increment */
/* PRQA: QAC Message 3201 : at the end of the switch statement                                          */
/* PRQA S 3201 L1 */
    while(bContinue)
    {
        while((stage < 4) && (rc == E_OK))
        {
            /* test if we are supposed to cancel */
            pthread_testcancel();

            switch(stage)
            {
            case 0:
              rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_SWI_CHNL], &Attr);
            break;

            case 1:
              if(mgr->sh->trc.d_cnfg.swi_chan != mgr->sh->trc.d_cnfg.actv_chan)
              {
                  pthread_mutex_lock(&mgr->ctrl.condSleep.condMutex);

                  /* Fix: when Trace Q is full, switch req is not serviced
                   * set this flag too.. as write task might be looping in check_q()
                   * using event flag has performance issue as it uses mutex/cond_wait
                   */
                  mgr->ctrl.swi_req = TRUE;
                  
                  /* Request WR task to switch IO channel */
                  TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_CHNL_SWTCH_WAKEUP_WRT);

                  /* wait until WR task goes to wait */
                  rc = pthread_cond_wait(&mgr->ctrl.condSleep.condVariable, &mgr->ctrl.condSleep.condMutex);
                  mgr->ctrl.swi_req = FALSE;
	          /* jov1kor-Trace Internal Commands
	           * IO switch request is from Send command task so RD task should go on wait
	           */
		  if(pthread_equal(mgr->ctrl.tsk_id[TRACE_APPL_CMD_TASK].tid, mgr->ctrl.cmd_info.tsk_id.tid))
		  {
			 /*Request to RD task to wait for IO switch*/
			  mgr->ctrl.cmd_info.rd_wait = TRUE;

  			 /* Close IO to get RD task out of read() */
			 /*(void)(g_TRACE_fnctbl.io_cls[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);*/
			 /* As switch task is waiting for trigger from the read task, let the read task come out gracefully */

  			 /* Wait until RD task goes to wait */
			rc = pthread_cond_wait(&mgr->ctrl.condSleep.condVariable,&mgr->ctrl.condSleep.condMutex);
		  }
		  pthread_mutex_unlock(&mgr->ctrl.condSleep.condMutex);
              }
            break;

            case 2:
                if(mgr->sh->trc.d_cnfg.swi_chan != mgr->sh->trc.d_cnfg.actv_chan)
                {
                  rc = (g_TRACE_fnctbl.io_opn[mgr->sh->trc.d_cnfg.swi_chan])(mgr, NULL);
                  if(rc >= E_OK)
                  {
                      /* Close existing physical channel */
                       (void)(g_TRACE_fnctbl.io_cls[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);
                       /* Update active channel info */
                       mgr->sh->trc.d_cnfg.actv_chan = mgr->sh->trc.d_cnfg.swi_chan;
                       TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "IO channel has been switched successfully\n");
			rc = E_OK;
                  }
                  else
                  {
                      TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Opening new IO failed!!\n");
                      rc = E_OK; /*Avoid exiting the task because IO is failed to open*/
                  }
                  mgr->sh->trc.d_cnfg.swi_chan = TRACE_NULLDEV;
                }
            break;

            case 3:
              if(mgr->sh->trc.d_cnfg.swi_chan == TRACE_NULLDEV)
              {
                mgr->ctrl.cmd_info.rd_wait = FALSE;
                /* Wakeup dependent tasks */
                pthread_mutex_lock(&mgr->ctrl.condWait.condMutex);
                pthread_cond_broadcast(&mgr->ctrl.condWait.condVariable);
                pthread_mutex_unlock(&mgr->ctrl.condWait.condMutex);
              }                        
            break;

            default:
            break;
            }   
            stage++;
        }/* while((stage < 4) && (rc >= E_OK)) */

        if(rc == E_OK)
        {
            stage = 0;
        }
        else
        {
            bContinue = FALSE;
        }
    }/* end of while(1) */    
/* PRQA L:L1 */

    pthread_cleanup_pop(1);
    return NULL;
}



/**
* Trace shutdown sequence for backend.
*
* \parm mgr Pointer to Trace manager
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
LOCAL ER TRACE_backend_stop(TRACE_mgr* mgr);
LOCAL ER TRACE_backend_stop(TRACE_mgr* mgr)
{
    ER             rc     = E_OK;
    U8             cnt    = 0;
    TRACE_cond* cond[]    = {&mgr->ctrl.condWait, &mgr->ctrl.condSleep};

    if(mgr != NULL)
    {
        mgr->prxy.exit = TRUE;


        for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)
        {
          if(mgr->ctrl.tsk_id[cnt].exist == TRUE)
          {      
    	      rc = pthread_cancel(mgr->ctrl.tsk_id[cnt].tid);
            mgr->ctrl.tsk_id[cnt].rc = rc;
          }
        }	/*for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)*/
		    for (cnt = 0; cnt < TRACE_MAX_TASK; cnt++)
		    {
          if( (E_OK == mgr->ctrl.tsk_id[cnt].rc) && (0 != mgr->ctrl.tsk_id[cnt].tid) )
          /*SWGIIX-1260(SWGIIX-1258) Segmentation fault occurs at end of all svg graphics demos:SGX failed*/
          {
	          pthread_join(mgr->ctrl.tsk_id[cnt].tid, NULL);
          }
	    	}
        /* uninit mutex and condition variable for switch rd wr sync*/
        for (cnt = 0; cnt < UTIL_NO(cond); cnt++)

        {
            TRACE_cond_uninit(cond[cnt]);
        }

          /*jov1kor<30.08.2012>-CRQ-Begin-Trace Internal Commands*/
        /*Uninit the mutex if trace be is stopped*/
        //(void)TRACE_prcesscmd_mutex_uninit(&mgr->ctrl.cmd_info);
        /*jov1kor<30.08.2012>-CRQ-End-Trace Internal Commands*/
        /* close IO */

        /* close IO */
        (void)(g_TRACE_fnctbl.io_cls[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);   
       
    }
    
    return rc;
}



/**
* Trace startup sequence
*
* \parm None
*
* return \li E_OK    if successful
*                    else error values returned by concrete implementation.
*/
LOCAL ER TRACE_backend_init(void);
LOCAL ER TRACE_backend_init(void)
{
    ER            rc       = E_OK;
    S32           stage    = 0;
    TRACE_EC      ec;
    U32           cnt      = 0;
    TRACE_cond*   cond[]   = {&g_TRACE_mgr->ctrl.condWait, &g_TRACE_mgr->ctrl.condSleep};
    U32    evtptn_master   = (TRACE_SIG_RD_SELF | TRACE_SIG_WR_SELF);

    struct { TRACE_task task; TRACE_max_task idx; } tsk[] = {
        {TRACE_rd_task,     TRACE_RD_TASK},
        {TRACE_wr_task,     TRACE_WR_TASK},
        {TRACE_chan_task,   TRACE_CHNL_TASK},
        {TRACE_switch_task, TRACE_SWITCH_TASK},
        {TRACE_proxy_task,  TRACE_PRXY_TASK},
        {TRACE_sendcmd_task,  TRACE_APPL_CMD_TASK}
    };

    memset(&ec, 0, sizeof(TRACE_EC));
    if(g_TRACE_mgr != NULL)
    {
    /* PRQA: QAC Message 3201 : Switch cases will be exected based on   */ 
    /* PRQA: QAC Message 3201 : stage increment at the end of switch statment.*/
    /* PRQA: QAC Message 3201 : rc might get values < E_OK in case 3 and 4    */
    /* PRQA S 3201 L1 */
        while((stage < 7) && (rc == E_OK))
        {
          switch(stage)
          {
            case 0: /* sync BE init incase of dualos */
            rc = sem_wait(&g_TRACE_mgr->sh->sem);
            if(E_OK != rc)
            {
              TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem wait of sync Semaphore failed\n");
            }
            break;

            case 1: /* Read config from file */
              rc = TRACE_rd_cfg(g_TRACE_mgr);
              if(rc != E_OK)
              {
                  /* FixMe: propagate this err to the caller */
                  TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Trace configuration file is not present\n");/*agv2kor*/
                  ec.cnfg_failed = 1;
                  rc = E_OK;
                  /*SWGIIX-568 -agv2kor Store read config failed info to shared mem*/
                  g_TRACE_mgr->sh->ctrl.trcCnfg_readStatus = (S32)E_FAIL;
              }
              else
              {
                 TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Reading configuration file SUCCESS\n");/*agv2kor*/
              }
            break;

            case 2: /* Open physical device  */
              TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Going to open IO in trace BE\n");
              rc =(g_TRACE_fnctbl.io_opn[g_TRACE_mgr->sh->trc.d_cnfg.actv_chan])(
                                                            g_TRACE_mgr, NULL);
		rc = sem_post(&g_TRACE_mgr->sh->sem);
		if(E_OK != rc)
		{
		  TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Gen3: Sem post of sync Semaphore failed\n");
		}
            break;


            case 3: /* Initialize condition variable and mutexs */
              for(cnt = 0; ( (cnt < UTIL_NO(cond)) && (rc == E_OK) ); cnt++)
              {
                  rc = TRACE_cond_init(cond[cnt]);
              }
            break;

            case 4: /* Create tasks */
		/* start RD and WR tasks */
		TRACE_set_flag(&g_TRACE_mgr->sh->evt_id[EVT_COMMON], evtptn_master);
              for(cnt = 0; ( (cnt < UTIL_NO(tsk)) && (rc == E_OK)); cnt++)
              {
	            rc = TRACE_init_task(tsk[cnt].task, tsk[cnt].idx, g_TRACE_mgr);
              }

            break;
            default:
            break;
            }
            stage++;
        }
        if(rc != E_OK)
        {
            (void)TRACE_backend_stop(g_TRACE_mgr);
        }
        /* PRQA L:L1 */
    }
    return rc;
}

/**
* Entry point for Trace BE
* This routine starts up Trace BE(ttfis) process
* and waits for SIGTERM or SIGINT signals for termination
* Upon receiving termination signal, does clean up and exit
*
* \parm  none
*
* return \li 0  Normal termination
*/
EXPORT ER main(S32 argc, char** argv)
{
	ER			rc		=	E_FAIL;  
	U32			cnt		=	0;
	S32			sig[]		=	{SIGTERM, SIGINT};

	struct			sigaction sigact;
#ifdef TRACE_ALD_INTERFACE_ENABLED
	int			err_tmp		=	0;
	uint64_t		tmp;
	bool			exit_loop	=	FALSE;
	int			nfds		=	-1;
	int			i		=	0;
#endif

	/* Check whether any other instance of trace_be is already started, if started dont proceed. */
	/* Fix for SWGIII-2024 (SWGIII-1920) MUltiple trace_be should not be started*/
	/*SWGIII-3516:TRACE startup with Invalid configuration fails */
	pid_t pid = TRACE_check_process(TRACEBE);
	if (pid > 0)
	{
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"A trace_be process already running with pid:%d\n ", pid);
		rc = E_FAIL;
	}

	else
	{
		/*Demonize the trace_be here itself, to remove all complications*/
		TRACE_daemonize();
		/* initialise Trace mgr & setup shared memory */
		rc = TRACE_start();  /*  SWGIIX-2071: CommandLine parameter not recieved in trace be */
		if(rc >= E_OK)
		{
			memset(&sigact, 0, sizeof(sigact));
			sigemptyset(&sigact.sa_mask);
			sigact.sa_handler	= TRACE_signal_hdlr;
			sigact.sa_flags		= SA_RESTART;
#ifndef TRACE_ALD_INTERFACE_ENABLED
			/* init semaphore for termintaion signal */
			rc = sem_init(&g_semExit,PTHREAD_PROCESS_PRIVATE,0);
#else
			event_fd = -1;
			//Initailise an eventfd for communication between other TRACE Thread and TRACE main loop
			event_fd = eventfd(0,EFD_NONBLOCK);
		
			if(event_fd == -1)
			{
				TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"ALD_PLUGIN - Init Failed to create eventfd in trace_be main \n");
				rc = -1;
			}
			else
			{
				rc = TRACE_ALD_interface_init_ald_plugin(&(g_TRACE_mgr->ald_mgr));
				if((g_TRACE_mgr->ald_mgr.epollHandle >= 0) && (rc == 0))
				{
					g_TRACE_mgr->ald_mgr.eventHandle.events = EPOLLIN;
					g_TRACE_mgr->ald_mgr.eventHandle.data.fd = event_fd;

					if(epoll_ctl(g_TRACE_mgr->ald_mgr.epollHandle, EPOLL_CTL_ADD, event_fd, &(g_TRACE_mgr->ald_mgr.eventHandle)) == -1)
					{
						TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"ALD_PLUGIN - Init Failed to create eventfd in trace_be main \n");
						rc = -1;
					}
				}

			}
#endif
			if(rc >= E_OK)
			{
				/* register signal handlers */
				for(cnt = 0; cnt < UTIL_NO(sig); cnt++)
				{
					(void)sigaction(sig[cnt], &sigact, NULL);
				}
				/* Getting command Line params  */
				if( (argc >0) && (argv != NULL))
				{
					rc = TRACE_commandLine_param(argc, argv);
				}

				if(rc >= E_OK)
				{
					rc = TRACE_backend_init();

					if(rc >= E_OK)
					{
#ifndef TRACE_ALD_INTERFACE_ENABLED
						/* wait for exit signal */
						sem_wait(&g_semExit);
#else
						//Close up Connection to host in case ALD not yet started.
						if(g_TRACE_mgr->ald_mgr.system_security_level < g_TRACE_mgr->ald_mgr.ald_resident_level)
						{
							(void)TRACE_ALD_interface_handle_ald_event(g_TRACE_mgr, &(g_TRACE_mgr->ald_mgr),FALSE);
						}

						while(exit_loop == FALSE)
						{
							nfds = epoll_wait(g_TRACE_mgr->ald_mgr.epollHandle, g_TRACE_mgr->ald_mgr.events,TRACE_ALD_MAX_EPOLL_EVENTS, -1);

							if(nfds < 0)
							{
								err_tmp = errno;
								if(err_tmp != EINTR)
								{
									TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "epoll_wait() failed in main !\n");
									exit_loop = TRUE;
								}
							}

							if(nfds > 0) /* epoll_wait() returned a proper handle */
							{
								for(i = 0; i < nfds; i++)
								{
									if(g_TRACE_mgr->ald_mgr.events[i].data.fd == g_TRACE_mgr->ald_mgr.file_desc)
									{
										rc = TRACE_ALD_interface_handle_ald_event(g_TRACE_mgr, &(g_TRACE_mgr->ald_mgr),TRUE);
										if(rc == E_OK)
										{
											TRACE_SYSLOG(TRACE_SYSLOG_NOTICE ,"ALD plugin message sent invoked Successfully %d\t %d\n",nfds, rc);
										}
										else
										{
											TRACE_SYSLOG(TRACE_SYSLOG_ERROR ,"ALD plugin message sent invoked ERROR\n");
										}

									}
									else
									{
										//reset the eventfd
										if(read(event_fd,&tmp,sizeof(uint64_t))==0)
										{
											// WHAT TO DO HERE
										}

										/*TRACE_signal_hdlr posted event. There shutdown gracefully */
										exit_loop = TRUE;
										TRACE_SYSLOG(TRACE_SYSLOG_NOTICE ,"TRACE_signal_hdlr posted event. Trace_BE Shutting down gracefully\n");
									}
								}
							}
						}
#endif
						/* do clean-up */
						(void)TRACE_backend_stop(g_TRACE_mgr);
					}
					/* uninitialise Trace mgr and release shared memory */
					if(NULL != g_TRACE_mgr)
					{
						(void)TRACE_stop(&g_TRACE_mgr);
					}

				}

#ifndef TRACE_ALD_INTERFACE_ENABLED
				sem_destroy(&g_semExit);
#else
				if(event_fd >= 0)
					close(event_fd);

				rc = TRACE_ALD_interface_deinit_ald_plugin(&g_TRACE_mgr,&(g_TRACE_mgr->ald_mgr));
#endif
			}
		}
	}
	argc=argc;
	return rc;/*agv2kor:- if error happens return error value instead of E_OK */
}

#endif /* #ifndef TRACE_DLT_EXIST */
